## Subresource Integrity

If you are loading Highlight.js via CDN you may wish to use [Subresource Integrity](https://developer.mozilla.org/en-US/docs/Web/Security/Subresource_Integrity) to guarantee that you are using a legimitate build of the library.

To do this you simply need to add the `integrity` attribute for each JavaScript file you download via CDN. These digests are used by the browser to confirm the files downloaded have not been modified.

```html
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.7.0/highlight.min.js"
  integrity="sha384-d8KTBLFjg2IVNtohjj32TAh8+MOFPukj5WKbkfk83mLz1ctxujsIN0jnKaRw3rW7"></script>
<!-- including any other grammars you might need to load -->
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.7.0/languages/go.min.js"
  integrity="sha384-Psy4WYRdpX6eaL0Yrtp3XkDq0WsaYTXresPU7qP7BhNpFnprhtDs0er4Q+fpiT8m"></script>
```

The full list of digests for every file can be found below.

### Digests

```
sha384-d8KTBLFjg2IVNtohjj32TAh8+MOFPukj5WKbkfk83mLz1ctxujsIN0jnKaRw3rW7 highlight.min.js
sha384-gHVRIRc+fz2U8u4/ZIMnHEkJvV45pOGqIoYRGid7r0ARA8p4pK1YnY5SVjP1lKZI highlight.js
sha384-nlhHnb2S52VMfEjeE6nTPbOfxJ9+H0Lpluvhz9sND4iwjYR+MYEAcmt+GknfYccD es/highlight.min.js
sha384-lz9sXdfT/Z/y8pS/YxMUReSYZfjcnWk1Yr0XIjse3VnKEvshTGQWjyroXdOPnc6B es/highlight.js
sha384-nlhHnb2S52VMfEjeE6nTPbOfxJ9+H0Lpluvhz9sND4iwjYR+MYEAcmt+GknfYccD es/core.min.js
sha384-lz9sXdfT/Z/y8pS/YxMUReSYZfjcnWk1Yr0XIjse3VnKEvshTGQWjyroXdOPnc6B es/core.js
sha384-1JBlycV1IO/WvwJZBg9Kx8af7QgXNgR6zM0Xp9mtnQ9f/kabBmfnsD02q75i2a17 languages/abnf.min.js
sha384-e4F/W5qSJkBE2i0gOc5+TlBW8vXBnQeCrwkPSqrN3mq/fW+QEbMMVwqKGq9c25NZ languages/accesslog.min.js
sha384-FlMPfMhv73PUOnh81YhQSFa2w3KP5c0u4bl13uGH0yEglt6tVcQHHfX9ivKFtZ0y languages/actionscript.min.js
sha384-bcBkGhZM/BEcxV7fuzrZnAWtyrMyMbzjTnjW1GdT2lOm5LXcPmajRWCZJLXRxGaF languages/ada.min.js
sha384-r6gwrzrP4QdjUQHU3T3ltUDaTl1DwmciLorASLN/sWa6quKQNTNaGbU2mqNoyzDr languages/angelscript.min.js
sha384-S826Dp6i4sU+5oogQwG89At+hEmQNktbSmCRTK4fEvnOjzEkrtVkiKCuqiSqResl languages/apache.min.js
sha384-mNq6e5qolIxK8R0tezKKbZKzn7yAtw0bbJPdpC/gQIskupzJn6Rrcejyv5EUwrza languages/applescript.min.js
sha384-muPeOCw8a4IiaRuNSFqqAMMXv1P6p9zVqefBN5K+3GHRR34KdeFYahdeQzbUifkr languages/arcade.min.js
sha384-CARCDeNM9/T1uVHtW7Nn/Hp1UkltToefBqWv9Ehoa2rnuLZrPS2au+F1kp7Cd0gB languages/armasm.min.js
sha384-AUbgDNA8vddTq5Fl43obRQku77bRZfimC8ezbK5AD47LpP5X4+7tViioefyT24ql languages/asciidoc.min.js
sha384-HcdOZgrXkYbACoFFgKYY1irXWh8Sz4jTdkMzCsNzA2HNxnei2EhBylWkEi1XOvnz languages/aspectj.min.js
sha384-efmS1tEJbYPB42jibSEFWail2vwk1ZR5Z1pZafcPlM+7U8a/qBV/4zXCpUea1SiJ languages/autohotkey.min.js
sha384-b0t3WcHgkhSb1IFtiuFSWffbndLy+PW6WYfYLwYFdFuWoCv+Xtjmy5SJ3uyBLxPL languages/autoit.min.js
sha384-+YmVvylYzUMZ3N8OfTVM/p7/bgChy7CBlgPubEb5Jh5bmQ/JK0O8eSBqMXm2EeZ7 languages/avrasm.min.js
sha384-lt3iTcx1tkTEXVYOBzqN58wkw9WT8trZJY/mPogrd9RXFsQdccbH6YhLMGEzxzC3 languages/awk.min.js
sha384-VApuO9c6wcqI8XYZu/PEbt8dyL6qmg4dbW9GwPiA3db+XwRipieyIhSlsfCymVxn languages/axapta.min.js
sha384-SWx9RTNu5IBJIlUAP+LjGQNeBAdjq3AanHcmCyPOYtsuLDQYo4ECK/yPU/nOyDX0 languages/bash.min.js
sha384-cq7JqfxNWrsxtnpxdLv0yGfprsaR/HHNpwPULgP8sls9gH+t8CEfrGXJNzeS8DkL languages/basic.min.js
sha384-osoT7UjAgNF2NJwFII4shufToUWFrJqBJiZ77XwG0vIcNXpD7Eu7tmI6Em+Nv9mw languages/bnf.min.js
sha384-Im11AKpAGG0UtOhdd7ymP/5vivcEFRstC/W+i2qNPwXxsP1QJf/xQAmT26i2jvi8 languages/brainfuck.min.js
sha384-4dlI9xQH8B23nMuB1RJQQ2UK5JTlPO4vk7/46P2mUFfzhu+hG2+T8110otzcLPhA languages/c.min.js
sha384-4Yyvpc+xouHUJKQRUqF9uVlXv0QazjeHoeaLR77Xm7dvjnPihMDw5/U1VpsCZSmK languages/cal.min.js
sha384-VLr66fX9qq0suZMiofgmacYrl+j0JBcpajg7X4yR3PX21tnPIggZspSfMzC4vloN languages/capnproto.min.js
sha384-cK+5SizVzM2WqRBd67zaqDIxNrKFoO7dQN8TAI3QdVAlNBXPZ45Nk2nqL7/uU42b languages/ceylon.min.js
sha384-3RC74l+MwFr4M4ig2XfbJ1WmfKj//7FyDWDte5HtV+3EZd5+DjE34CgR8Q7IRkta languages/clean.min.js
sha384-tnxwUBTXPh19YiF1IfzUx+Kw4sKAeDWpVoQNN1RQ5dKTnD+EgGWpR00OqCP9NYpJ languages/clojure-repl.min.js
sha384-8JWkp32oWTEVxxtNrzFvXIT0dEfL3iIzNnscjiUdgvFNq4G2o/kfRsL6fgojtVZ2 languages/clojure.min.js
sha384-1VjLXMtJqNs53XCu/2vEaKLIhqVaHTUiJjwUmsrH/9vV9N8++jRXIC/n5XmCTGYu languages/cmake.min.js
sha384-GzsGY69vk2UQjzojbiZGp1H6epXSdyxRIR2m0W0rZaKM0Pr0C6jyb0RoT4w7bcdG languages/coq.min.js
sha384-vzwcB/AGUR8E8NYGUap1vHiXdnu9MvrG9OcGpPgxiMsTl/UTisxolNgriL88V+bC languages/cos.min.js
sha384-jeDC7ez9MZKKNLI0oJVaj9Zt6QWOTbD69feJDuYGCsIkFXS4d0U9/Rx41687+6am languages/cpp.min.js
sha384-09x0O3i7ldnO2erjJZ7Csq5Ly34ynSgg302rNapOCM+B2NTXUbnMZK7CAw3StHKG languages/crmsh.min.js
sha384-ez3Hk3Hg9Qqm79yJmFV9Nr7Y6daks/jkErtPeYL2bFJ/iRZvBy1NE1PjV3p3S8DK languages/crystal.min.js
sha384-Gpy4bdPSCx57yFTM+1RjbJKj4ltJhuZL9fO/bLCRXjhb5mCJ1SMmKDIbzubGzjqV languages/csharp.min.js
sha384-rVq0TDgdBV79IpW+nzS4jCEavwE9D+DdsKHxmvseSaJjMwZaNokyoD6/flk21n/j languages/csp.min.js
sha384-4clploDvMqipq6fOWOVSpq2Vdv1PNDXNUCDc6aO7/IgDPALgRkocjxNfbyZtjjkp languages/d.min.js
sha384-kCBQ3kQJRNsdWCWsOKi08Bjgd0ESBgbf8O25xuuKMWMFUf0qRr+PTzSzgiYrDpzE languages/dart.min.js
sha384-FBte953TICDaFp8w3aTtxag82WosBt4JQKCtqf+io0KVggimNKMob5PghzzwxbMH languages/delphi.min.js
sha384-BGrtbSWkIGEbzlKDx7oWRNsdgEf+oKZqGoTV5WjVXOBGeBf46Z+YdW5FfE28LCCc languages/diff.min.js
sha384-LtuxfUZDqHnNbdHSXkOuKLYeLmJzmdx6ANxrVw6cris2O2XX8Nmx0YSNTvq95AUz languages/django.min.js
sha384-+nrxzzUI2p35wH1nm/6DEpE1DCCYkKjfM6t2/2xh3qw0po6txqhd8s5stBmZBTJ9 languages/dns.min.js
sha384-5TrkTRj8EM07q4oPGiK8VM6JA9PBrW9wqhx6LOFdeNikSVzS1EPWbUlWLWL+R/9h languages/dockerfile.min.js
sha384-THY6zaVkBNlc1uCwTCsjn7p4NdGmgvXUYdMoH+lZstS7PL/U/PXBO/lqRbGy6S6w languages/dos.min.js
sha384-geX6kyiGrL1scbYmPV4+wgEqVPu+f1MXJA1lAPxJF1E8xVbbL/uf3BqOT2sPeM3l languages/ebnf.min.js
sha384-ha0qxgqEGMMIQKLzHuMBwubVuvZ8/yd7RdMy7h10LT3Vf7dm11/GPN6KYZ4SkScJ languages/elixir.min.js
sha384-WFIUIocLL6tNQ4j+decNLFJda+Y6tbF8RZwI2YKFGuh69TPKGJJwhX/nZ9hQp09r languages/elm.min.js
sha384-LFYsnzCd/In7kZhGVCrYyWvCmzKOdi0I0LZ9ig0rX13MdVwNfnM69o9qRRFVLUFW languages/erb.min.js
sha384-EHFwUBCXxtqKEwWUS2uw+ja3kiENhoKymFh0PvqN23LR/XTy2HKDVT/3/wqYCR57 languages/erlang-repl.min.js
sha384-A5TYKVUUTDfxE+s+njJjCZrL90jRNNjVS7MeGBpsbonDwZN8VcrxQ8DvWaJJzR2J languages/erlang.min.js
sha384-Vci3+kHoG63J7CztWtNDzvgF2XLosA1CvsoV8JGMQkJRUysLPzonPHOoc+GJBPsj languages/excel.min.js
sha384-uNhk0Fxu8cfwlI5md/AsYntmRCTlzyYsoOFA5GL36C6XQ2vKvjrrMuyUMi5IYWkF languages/fix.min.js
sha384-HqBA9hI6yZf3xhn8123oKbE0dy6yTQCAM/O5lgjz6FEilUGhHMp3oqcXp5/52K4W languages/flix.min.js
sha384-Buy8e5V6AcZMYOqnDi81pyvCqCZBVknDlu04H6pwr7xkSWwtFWoY3BNjgT74/wR0 languages/fortran.min.js
sha384-dtnrVGjPni14Lwquxdh0ezqYkwaovAdgPuL3AV3kAtQQAHMaSQyrHLuxjKDWt+Yc languages/gams.min.js
sha384-/xVjWh31UaLPGoRmOOgAbOMFDlQH6vNwnjnyArjltcv22HeaNFPc3qOqdQPg8iqj languages/gcode.min.js
sha384-J6PyUnGXRtUq4C7C/QP24oL4gY3Ff5cW9LIqZgxzfDhwbbHB6p1Oydk88hE6rMiU languages/gherkin.min.js
sha384-P7x99V4NjNi1qHP6nlDB9jFkCt1fuPsEPoGWzVb7U9YDnq7i590gVKV/gvtWEk6E languages/glsl.min.js
sha384-Psy4WYRdpX6eaL0Yrtp3XkDq0WsaYTXresPU7qP7BhNpFnprhtDs0er4Q+fpiT8m languages/go.min.js
sha384-ixnbBfBvgUDRZYmzE3fWKbaY+Ro15lXxst+rNqLJeFLuk0D7DnYr4mIOhbwstF7F languages/golo.min.js
sha384-6zc7dfiwk0NBhLYrSrFrCP8UssyJO+I2zgJHORQeDvfWrzhMFrfN8/yK/h2Sr/hj languages/gradle.min.js
sha384-DkNotVJcK5GVqF4TCIDQr5Rbgk0loR/yuPNdW7NBZGydoYkPDRC5blmv6h2oHpau languages/graphql.min.js
sha384-bQYV0wHKp+Nm2pm4RQgWcc4+GLW7ATg8VwUM0QUrJ9oDb2QXP8YbJ4PDV+tcBbxC languages/haml.min.js
sha384-9dupACs+p9kaSey6BO0V2FzxQ2ildDxzimarMJOGZEUl+Wi7mZSwxD8eHeD9ZgZq languages/groovy.min.js
sha384-eGSs100WCR2HEAR8MoSDdcALoQxoVL4SYKj78ANn0AmlWUw+U7K6ADVAgeVfw2Wq languages/handlebars.min.js
sha384-79CckMIS/53CIAfoXYNgd9QOEiM92p6MqdXPU4sb+tHagf7VF5pJJVducFmOL2hK languages/haskell.min.js
sha384-sqFRSdgZAmGKBkRTqnPySlV6OS/C5xJTAyh61Q6obtyTL52miEtgabkmZlOUhfJN languages/haxe.min.js
sha384-8KzZftUjNltvrArmvJhI+6MrAvCycQLmsVO/MHzvAkxG08qjPD8y+uP72xmMSNfd languages/hsp.min.js
sha384-ssyIa3zIw8JomlumtlkegK2I2cGZ85i5Tn7vSwn5LViCRsolBCQ51uLcrsb8rSvs languages/http.min.js
sha384-jS1tuhAb+1LoMwWyFwT5xLxwlCT7RNZGQ/4o3o5XsddrP8ygIoI3kXPcGSvl5HZP languages/hy.min.js
sha384-ipU2SaunZcJ6KU24PpXAX85h8sWfvEXZ7mIVX+DuGGjRcDMEJ/Vk2sgjfo2VBaXN languages/inform7.min.js
sha384-nWYjiLUNS+Vo3lFs1brpQNVh3EaC+kHlkDgcoxIHrx2rTkb1uayuxoxKKju9ejDx languages/ini.min.js
sha384-tk8QreylLSzfZy6S+zyVNFurmj2/kRrshqUP6cw7MfwbezDnYenRRXRmItqZrAYN languages/irpf90.min.js
sha384-0uAV4Ud+faOeZPfS/5D8RBwJU1gZCI8463kUSvWscvrt0SCPfC8PZ8Vf1b7A9wsi languages/jboss-cli.min.js
sha384-iobZTYVzyrsWqW32ac+XJyMw7EWFbPi3u7zdOpEOVt/L0uR4Rfnt7UBD7cRYq19G languages/json.min.js
sha384-fT9i0rg0/hRaQ4jFh0zMACUO3AaSryLkqm2J2M6RfQbq4OmfSXHfXKW1t7JspYZf languages/julia-repl.min.js
sha384-c6KqDGT0LCeM3DWCwlolFxLOn/ofHhAL5IRXRh3llC+nGRTXp49lbth3MMPbzYnq languages/julia.min.js
sha384-JJMEQWBqOoO6mVGAQCpn4zfXqsdnZaqemaIq8iPQzdz/Qw5dqqcsz+u0vp0O2so7 languages/lasso.min.js
sha384-LoBb47ALy/C0u+Dp9Y5Xw7NXFVA9wTY48ZA7my3KgPLS7cbfdJvMED5OtxOmAyXP languages/latex.min.js
sha384-kuoxvHEK7fD5RMzs40b5GH00FQ3Ya5bDhycIprK9ARihZ4yyalpVgHGaEtRmfWgn languages/ldif.min.js
sha384-whORxn7h4Utsz+G71cUkEt1onjCHBeOaqwJnm7wUStp5YV0AsjYrCamdLN0uQjVk languages/leaf.min.js
sha384-xWUwdYgFmJGcGDq+EtCdrYLilZtWBPkXXOVP97sIo/0fdOVhPn49N7lGIPyVL1Ec languages/lisp.min.js
sha384-/aRfsgTDLoM/p+RIzQjaeUMNLfZfYUWeLrh0sLT5nrOK5wQbJIyltPdPNPvteOFy languages/livecodeserver.min.js
sha384-zu0OnOkeecVgprQ72Bo/7UbRwqUmFUmq7wpPqDW9B0QLXCEZepMmHkxmRV+PI0D0 languages/lsl.min.js
sha384-3gwLzUwfIeRcymbHpxfaA2Hq4BusPPLiD094C+2Rr2ZT/50Sr2CwH7Jw3e9uD7rZ languages/llvm.min.js
sha384-fHXVfUUIWqzYK850yZqZb2gGcFgRbTr/LA3xiuRmC+xEjxQ2i65vLi2kR6zI1Eky languages/lua.min.js
sha384-MonxLgrrv1RP51AgieZPeAbGdZ8BG1PObLsq2xxlOI7B3wGjER0TLc+MeoQ5Tqi2 languages/makefile.min.js
sha384-ygFwl5Lw/eHzXz4hhvyobzWIkQzLobd1sTOfRQL/l5AX4xY1RlqgjdmodNH8ksxm languages/markdown.min.js
sha384-6OhKR0cONBFZZAe1U2NVTsavk0jJ+ppjGqsY9q5qVuHhqlQQHdF72gLTKgFBkpU3 languages/matlab.min.js
sha384-6bIDjqKeriXdPdqvZxIUWuOmrPPTHVU9Pu+n/S5bZ96fUOpGLBL+KmFEINL76Ela languages/mercury.min.js
sha384-aaCjpkl1UQy1o8zyEa0/oYQ6reKDZbt2Hs7vg8aUg8FHtrYbyC3X7la4RQ6zC/GF languages/mipsasm.min.js
sha384-yBd2d8omMuXxBj6X1aTsDemD7faVcdFdhBQekXCbrkUwdYgRvOsxXQoVVTNcZm7i languages/mizar.min.js
sha384-7Gs1wsO5uzpO9LBASmTMV12pp7uaZ3Tb81ovfR6I/diFt0691+qbgVQuY93eCmj4 languages/mojolicious.min.js
sha384-zY94au57F0VBqwVKCeU9m5p6QN7Cg8CprbcmmB5XdYgOWY49BxjX0f6FM7XKwtrO languages/moonscript.min.js
sha384-L1exwpbsIwbzdcG5s/8FZiXhaAOoAFidowXnQr1qSuxYJD+vJmTv78XbBPGjQO5T languages/monkey.min.js
sha384-c54kBbzgux6OfWydo/gFB7aqMBJDAhIkXv2KHgyEdlULxTeOcTJNS+lVmuyrNpYh languages/n1ql.min.js
sha384-Rhfecp2PVwMeBwzCzSfwaBuFLGhv+ozHo3b8DNNoJBZciE4rHDX2WudFmZXSrqRB languages/nestedtext.min.js
sha384-dOT7KM4RtO3aID6IVcP9VlmVOPYv7uDOOYEmoK9G3RSUpuk1WTtlw/s7xMEaUrIx languages/nim.min.js
sha384-ICsb/7D8sJ24rGIr8Ibk5m4S1HcJIVtcormR514tuOt2FEEkRBnmwTGT6XrcWGet languages/nix.min.js
sha384-V5sgbljhgXFNHGd5Tgoc4bDpggomCkDdjomNVJtV7/fRV/p1SJj7FZb3vZHKQsGo languages/node-repl.min.js
sha384-PwpPry8IVDzDPQbDCHEU++BvKczCDFO3TvXUw3ZEmYqFyuf4bNHBoFoXJYiYZuRf languages/nginx.min.js
sha384-Bb60gbAQA679RUeGv3Qk7fSiOj/o5TYzKL6C2jfdduydFsnBahq4prZMXQrtpVIx languages/objectivec.min.js
sha384-YzGPu0sbba8Gv5J9i3BnDAzaU0dWoR2Todtw5y6tYPAJexDKraZJUtkYJvmJwwtI languages/ocaml.min.js
sha384-V3oj4PewyouNQFMZP/yl9tRDWgNVRO0A/Pfmt+EB/1JG/Kege6hZrNqh8efGYQ2w languages/openscad.min.js
sha384-sU5fc6tjhBKdeoWu7/EU+7my7lMDSMJmJ2oiI1yKrjsUWaXR9PI/UJa6JMg2pNFF languages/oxygene.min.js
sha384-vLxJxipDJKnu34gz2jGqLhAx7u28SWyd9dEKNHwZO1LfaT7W2cPUNP9M+8A+MH0M languages/pf.min.js
sha384-smp8JMdMwVqu3up93nrY93YMu88g1p71e9ZN3PMgIQUthixR0+iSLfaVoXVu/ix6 languages/perl.min.js
sha384-lUJDoitCa8OzSK8It8SKMKtOi48d+V8OX1VBoyF9k6WmOnJT3f8NWPc36X+K9FHe languages/php-template.min.js
sha384-5+OfP9Gp77ORvKpTnffrPU4eHNAcSUiA8ToumAzxfQMxfVZo6inB+kJBC6LOvq7M languages/php.min.js
sha384-cg1lgvzaGM6MnRZxsr9AttpHVfaASmpMf8yczesVpE2diEbWnmGZ2ehvGGrJ0i1M languages/plaintext.min.js
sha384-RBdLS+KFAFOrmFheVZzhklo/FaUIcYAQXtehuY6X2SAq2W/frmBXHef6ASw+jSwV languages/pony.min.js
sha384-wXaM4/99k72wTmGhrWCMYEDPLn74MgE81S9iO/xEgS5Zi3B4CUNjz5lvQFe1PzJ0 languages/powershell.min.js
sha384-hbOZo02xX1qWy7pPLBolPlrKJrgFrCI2f4n5rWgWEP6ofPwGoWB2tuVSqaKAZheB languages/processing.min.js
sha384-DkPRblOqo14WsP8rP+gMV8vLcRZCjtmJXoq0mX5F2EKbwxSC6Ebjbv+wO755h+8r languages/profile.min.js
sha384-uEwAXEqAs0hHsgH86h91M48nhFazq3/Ivq16va/5gGENSDH0GW6/o9dUxWL2f1Tr languages/prolog.min.js
sha384-WN6c0/GTTB4C1HKcveMmOUIYzlWEsMNwc0Qp1Fe/esBTkR6fI7m8/jbGSW+ofBJP languages/properties.min.js
sha384-kPca+wDOE2z4IKzzsFcesF8R5x5s33yOvRN6TPsRZfPNkdQ5Pr8U/H9Zas9LHOSW languages/protobuf.min.js
sha384-YhePGuixGK4QV7QPTjjlITh9vmtpNrjcVKAoObo5fMqxdCGui5/ms32jx8TModTo languages/puppet.min.js
sha384-V4xn4vj4301n5VRNn7Htld4684zeqE3r/kvmYxZRPQXoS/YH4m60s0/WighTOIZn languages/purebasic.min.js
sha384-ivl7MRk0kYpohcRlZTQgSOqtYx5JdfyGlPnZ4KiqzPKCh4iRv+2FeiV5rb3Es3N0 languages/python-repl.min.js
sha384-0lBZSfrCeXe5kt4mQz2ucFXCTAft4HXhXWVnVLRG9p0i1eJVDtglXaBQWk5p7+45 languages/python.min.js
sha384-+N6raccP6AGzdIsZ/nL2d1plNMnYfALZHGDH6+8YFTr0sDGnkv2dza1bFotuczql languages/q.min.js
sha384-/mJYiBKeFL1VUJzxdiJbLMJc746D3QDRrNjocEFWHvgW+yOBpB+gKzKZANvdp0MO languages/qml.min.js
sha384-/XcxP7bhUHQ6R99Fgmy11Fgf89oallJ538V2im3yqqDOdpba8KLlO6l31j0RT6OU languages/r.min.js
sha384-DcpBmrRrQAwoPdYXVDEVon2JdyLZFtizl611RfTZZMdLVDMbtGWSqtHq5ST9Nx49 languages/reasonml.min.js
sha384-ZgHgAFZdphjpK4JppmqRUPikmsr8ogq+VCJPUvrK1WnfV+UPxh/0ed+O5gW83nco languages/rib.min.js
sha384-d3JjonjIySkAiaEKnMtMKuso4VPciq2QbY7K922bqLpD/+cXoJYJ6ZMuFFcRmF5e languages/roboconf.min.js
sha384-vUIelJWN36LUWwHb6mNd+wmYc9IwuBXMFeDMTzDWsd96IQbdLo5/C3yTaUyui8se languages/routeros.min.js
sha384-UOwRiGrKGMVPUd0a9lm/kUBZPCAEos+ruIYICMwEaZlDHIbbGrTNoz06BpMI9i4L languages/ruby.min.js
sha384-62CeLsRAF3y1w6bfn71FXQuHwAi5t+NfBaticApund7NnbNsqJYtJUnacTQIWNC5 languages/rsl.min.js
sha384-nsHpV92PIAILuFaoKZLe+jDmJE/ER1/TO7vIfWGKotDkmxvAKLhFkIhareRpd3x0 languages/ruleslanguage.min.js
sha384-KFQxGnYz36i1Q8c3QUrdXK+gPP8r7vyjhntYx/YOWZzzc+/xVw0XHfZRjqVV5qNz languages/rust.min.js
sha384-EDMA9MsgBcYtGaM/uMs3IFZhQRE3WWG4iAsDEJlwzP5rI6/OjNt3IWlOr+mWrFJ5 languages/sas.min.js
sha384-aL4V34PqQVriBR8ATtOb1lJAmcaN30ZXYHb89tbVLMZ+NN7g49RW/p+GrIO22RIQ languages/scala.min.js
sha384-3zCxexEM7CC0BXh8dj2L+/mUitRmVUc7EIYJKGs5FzgcYZDpoKp9Ny6Gxy0R7WBJ languages/scheme.min.js
sha384-0ZoR67H1fjEbu0Gj1w1SPOEJrLLPnoFZCaaLXO4jvfy/rJMOXBqNEUdFMqGQsX3M languages/scilab.min.js
sha384-G/kZCApLiLnZwvoIbWQab/p0JPEI+X6Pxwbyi4FAQD+DV9Hd4qfd31z4Ayk53Ouq languages/shell.min.js
sha384-9jx0nhoGWnoc4XzWXIi4LhZ2C51BfhohbCqUEK8y8NVZj/xYuB+s+UYJhXYdaa5w languages/smali.min.js
sha384-6mdoMq+n/o7FGYHsC3H29GsO1huGQgpTo2D5859imKGStST2l+JhYOvukyr/2AP2 languages/smalltalk.min.js
sha384-9djaUQxDnf2rg0D6P+4Eanx9o1LTk72j2+96SHtAKS7ymqC8wsU8JddoFRywPTrW languages/sml.min.js
sha384-iCUAa6F3jkbdQ3FDOnbKUT4/0+OtueTIMjCP2UvDlQrloIwFwE1ZCVQ37VT7yZMJ languages/sql.min.js
sha384-6sxDuTblx924pvWMIEaxkhCncjORAJJztNC7EtOwR65qfTea/oRr7UCxkbMnIsB4 languages/stan.min.js
sha384-jhB2nw3/p7a+yPkejrhe79OFlCBTkGJvKcWSSgW5KPw73YeSxXOPtYesB2G76Qq7 languages/step21.min.js
sha384-zZaz5R/a3FPNMd29OCiuSYqYB/A7uIgnWrRP3WEgyA0boYQEltR9B2JwTgXTojUW languages/subunit.min.js
sha384-U1tz3lNBW4gSLjCU3ClFkIaz29SN4+GyfeoGbnycxnoKmgDjV08KptPCzmMEBzy1 languages/taggerscript.min.js
sha384-hZ1WW5w7x4BJGXvkvAtbI1rgFGx6eG9D3UCRxCVVhmF3C++01VJwCE7Dq2NfAtNS languages/tap.min.js
sha384-H5kVc216Z2fPQwu/x52gunIw3rE57MEGJNzqsl2ctLjPvF7zcwe58DBvo1U1zPWq languages/tcl.min.js
sha384-uDMUIDt/u3f5nT2qbufqhEnuFdWzsppKJiWbOGGDSLeKuzwBSTNubgf0WQpG2FUH languages/thrift.min.js
sha384-Ceixm0Uup39ZQoWKlJ3qCgCiHBSzAgCwV4VpEwjeCJwrhgoEFPUcyVgeDY7r0/tg languages/twig.min.js
sha384-/L0IcTMkVf+7Emkq1uJGUW+D/zIc6z38En71sexkrsaeR2pIrSoXW6+p06pSDJEE languages/tp.min.js
sha384-ABDAqbte/z3dkuvIZlzZGgH6IVImUH+BkJLyUXtqqLyRKvgVEznTDl/rkyFPaXuP languages/vala.min.js
sha384-n6wUTgXDFotIhmUNyVuRx/OwcPpTt58hzQl6pUvPY/K+YpZGtV615FEkLqcx/4gH languages/vbnet.min.js
sha384-JD7NRQE4aItNlS0ZsZxwGuGJ8fLMgZdGy/zLgsJCShPniouUE1ikIydZUERtkosh languages/vbscript-html.min.js
sha384-oj/bTFwyS+cCp57uOUVZ3XbAuIe/NXUKlgJa2BFpZ6NlC+U2IKJgvUIDfDurzf0K languages/vbscript.min.js
sha384-lnLie4IaUN5+UmOmYOGKnjVl11qbj6YDd+v/AN0Yht8uOPXvco6fAhiGRJTu7Dwb languages/verilog.min.js
sha384-uFtHnKgkUMPurMGPu/hf8OXkO0/Z3tEpUTGO2CDpkZ+9Duv1jFuPWbwFEdTEdwLe languages/vhdl.min.js
sha384-sTcBTpPfPQ+zTJGCHuwYZfs5+r04YxFZwJ2sRb5bkPQbLCIj+PgtqV1eM5DXfYWP languages/vim.min.js
sha384-BNpKC3AFKcFCohfjnfbsu1wWizSAy8y4SmZ0+Zhznt/uw2MNqRu30NL8tDdODnbQ languages/wasm.min.js
sha384-I6mhowbnu0ZMyJnp+uw4h2p4lOrsxUiw8A8wMiJCj5vW7lwoTHFbtOqfsXp0DeQW languages/wren.min.js
sha384-OTt5uy5kf+2CDfIxZZSVOZgEr7C7PQOzIMYb7CNfXOe82n4B+zN4JUwAEcCzOgNG languages/xl.min.js
sha384-QVZ+m2Q4pBQj3XpgpyE7kiKYY7Z/b55Xgf1hIpg/26PjfQtvn/YGYOt2B9F5eExw languages/xml.min.js
sha384-6xFNBPtFszho+m5kclphryIts6IP5dnCml7C1ocugYkPrjFJXTjv/8xv3UtCRMjW languages/xquery.min.js
sha384-E9VpZq+poGRLEoyGHF52nKOUzQSPGQ6oRmNk96OLOhfQK7iQqFodh2oAlntRjKqN languages/yaml.min.js
sha384-0jW+Q5nwV/G/DNMX7FvDjNmNErVP0pwENrw+DitI0MsE+/TtX3z5fIabYvbDnYBg languages/zephir.min.js
sha384-zrCx2ZQcmZM+pIaUhnKZODDHPjJ20Yj8G9FEEJ4ZH3xscMBFoItXq02OrseH0bRo languages/parser3.min.js
sha384-ogL4AWEzvsyO2uwKIoyFRzZWKdxbDiCqsPVayy3gJXWkR1y4feL9MwOCygJidZmQ languages/gauss.min.js
sha384-z15ybbiAm6IlMTUcjYe8kzDqE41p2bPR7yyTqk4yNiZ2hnNY8QR0r+Ta6r6m0Cvy languages/mel.min.js
sha384-XiZUBZZ2khpwjrjCVQ63moPm7UFYqBS8i9FqoSisCOGcl4U58pxnqH+jSmXg4dp/ languages/dsconfig.min.js
sha384-Amn+HXvYwCZrGRf8bsJgb1gkcPxzNfcozmUt4Co7tHYEDwdV/ZSxbDKuNqeYS2OJ languages/dust.min.js
sha384-nzevQRuclIHdkHDUFOAWtmBkNdBEDmNz3UCWzA7P1y2FCzGhWwC0UkUBTmCSNH8y languages/dts.min.js
sha384-SBM+tKWcGjYV1hcfPEFLLTxLvnJenfFBuTXfGfxfucFXQMRrlE8KEwTa09Q4EG0R languages/pgsql.min.js
sha384-2U2Xa1wsjWvrwf8mFmsvAv3acSabP3eIG31/a4hrr16/yO9WiBDbpA9FghOiGhej languages/stata.min.js
sha384-KLO24NsmNyBP3gnsH9JExc1UQBMiIVV+7OwnkN2JklKlpv3g04Etjgca2ZBZz7eL languages/x86asm.min.js
sha384-0SW1OiZJ0WbwmtHBZuYzMHgbxbRYt3iCP0ocfM9uXIZTgQOqCknSFi6LthVbnZRj languages/maxima.min.js
sha384-d9DHAkp1Z3hDfmHa+zLuFWhWwWoF+hdxFYFApXUWKPLMEvmkr+c7NPuioj8d5+1D languages/1c.min.js
sha384-vZwGFxKqG7QOFGnLAn6xZ/U4bfQwZ1HIK31Y8h09Bi3NMfeYikp2T0UeQ32lRLjA es/languages/abnf.min.js
sha384-aH63dwOZCYcS4UNHF/W2asQBjVZLzRCo1iVqz25sQITBw8Es+CyuFK7GepiCBj7Z es/languages/accesslog.min.js
sha384-9BbvtIHrlK0SavjYk0LKm2cnc2gAqNFiwTKnauDFjdYQqVYtPleWxwcVufi9jJ/Y es/languages/actionscript.min.js
sha384-r46gCx3kjd7Ebpo+8tSS97XnWZVvaKzIkx/7pvH11SyVPqESMUD8h7HH6R+I14LX es/languages/ada.min.js
sha384-0nAbmVrX7qr5cCvdX7QU4gQX7RF0loKLyNuQm+ZbsTqMwTAJ/OuMxGmW4rlNi1yN es/languages/angelscript.min.js
sha384-0gs6itOMKpmlv93riwHpx2EVR9ZG9jSxCqI6HcTHJOt8fZV4Oihpp5mo0/yjR57E es/languages/apache.min.js
sha384-tLHzQFerZTkmgoftYwDxTp221K5//HJlLS6gRXXQsSy8L8vSZJnQ0872sZ5lHv1a es/languages/applescript.min.js
sha384-BMj6TMBSxTn4+yl9Uh2C6HbpUEmq6TkqZnUGmBvgX8H9rFiwWHa+nqJWVNoQoj6R es/languages/arcade.min.js
sha384-jMFtMb6kf5WMBEWMHXmFRNErGM6Qc2rCMwpQqPbTCh7vi+8scZwTduTcjYDMZJQS es/languages/armasm.min.js
sha384-FtVVNb3k6VUYdcviq8zbYUSckhQmWgJyiyF3xhbPrGxN5eXwzM2zE7rkTA1bhyol es/languages/asciidoc.min.js
sha384-1sczkTpp0wlvaGA+RhXo9SO6ACv3j8fWjjg7mUfg3qWf7T3QkLWrX5jHyftx1lLo es/languages/aspectj.min.js
sha384-4k8JrTANx72uGfaZ/u26deknlPIW99eTw394+QyyW3gFO/kCH9ZUDtMfQ+xIDMFh es/languages/autohotkey.min.js
sha384-2cFBQVnPEFeovnSd/ZvZ2i+igk0t/JreZdMNovOzIMg75JL0L/yhWHALGMuiW85E es/languages/autoit.min.js
sha384-Ch8xyZ/9AFka0OkcydOUJ35jAoeHWTCEsPxTTy4CG3HdQtTtNd4cA7XUB+c/DCkF es/languages/avrasm.min.js
sha384-5vNTohuBxYx9jAkDyKlZZ5io008pOEZ4OjfYeO8Abnrrl4mqc/efcA9jvYtptL3U es/languages/awk.min.js
sha384-4ZNYpNMyWYQLmnOfs9jXwJZJPrOX5owVtD/dOMOXUfTuz9B9DcKUjwgmRz1ozIfY es/languages/axapta.min.js
sha384-vvGIMuuSefsyBMK7YEodOX+X3OBt4RK6dHNFYuFoTVQGTUGGGKa1sTx02Zqkcr+k es/languages/bash.min.js
sha384-DV7hyq5cMULEjFtnUO4xe/CCV5kYTb+SmRD+1ivQG/XmkLuKTuqlVOEMRZSGT3El es/languages/basic.min.js
sha384-kFPgzK0izWAarNbArxXDrBs/ynCKDaBmv28gWIOnw5EPToR/RSWBykwhZFdN2pO/ es/languages/bnf.min.js
sha384-5jcVU/7j+iEDizV7zWklEp6vuYC7T8nggXA5FZFfpW3KOtZYoYtB9rV4SPsteAbn es/languages/brainfuck.min.js
sha384-0ed9rCa2OuXQJKobbM5Use5VlTJqaUAi2qcmqN8r2ni6vxj/gmvOYUVxzUm1pTZZ es/languages/c.min.js
sha384-/9EemhgkvB5HBdH7XJOH0aw2KPmQcKxYi3g+MgbqYUJzG66tzoLebfui7kMuYcGu es/languages/cal.min.js
sha384-wBp8cTlpaEtM/2gzBoWUyS1+ojTx1c1bCJq/nSPzQy7z3xq8tMATdvP9wkJ2QAeP es/languages/capnproto.min.js
sha384-3KJ0snJxyZE888v2tFUObWPioMyVsYPqe3UNW4mECKyaWODDgKx8mbEPvVB6RrsV es/languages/ceylon.min.js
sha384-uwlghw2UXC63EWEqlxk5ZaCosWjuUb5yeudjHsys/D00P7wwdYG2XhwXUPEu1cpj es/languages/clean.min.js
sha384-pDnEpz4dJIsOuP3m0+puGGngdMZ/JSEb6b5UjTHFjjbK0bQrpvIYSo1fdAaRxpWD es/languages/clojure-repl.min.js
sha384-+fj91kiYzobMduVmOrbfom11bvMFMbtqzbTGXyOHRdrN4ygLTPTQMNrngdg+PVI0 es/languages/clojure.min.js
sha384-rkJ9Dhxf7UW16WPbPo+abcq7JhF+98ptVWsG8rlVi2+GENp3q9PD1foJnjLMG7sU es/languages/cmake.min.js
sha384-hzc/r9NGgay/fu/kX3DhOE6SvPGIl2Cha0WWnoly5hyPMUNNJ1VEesZgsFiTU1kb es/languages/coq.min.js
sha384-tBUVPy7+AGU7v4CEqm9GABarII03WIh6oB6xtWVttu5Fc2nA1RbnfBhxWiMlBZUE es/languages/cos.min.js
sha384-jo8T79clU4QxtKZxKjlJgYbjaNtfzzCxB2My8oMIrSfYCyotz8CI8k0gqQCEOcSM es/languages/cpp.min.js
sha384-Z4o/4rylryQekiD2SHRoS6ADGU/J/7Wvw7otPq3PFKmvK463NeDXCbwVnD/GxsWG es/languages/crmsh.min.js
sha384-r0O2E9zK+JWj1BcE8Xop0co1Thj6bWIadBMvN0FE67jBtomG1oHCa8bxq+1VytN0 es/languages/crystal.min.js
sha384-H783T3/V53bK3Cc7aH/FsDJrXlIhBMWdTVpusatsYcqLE4kZj0nPG0aLY284iC5Q es/languages/csharp.min.js
sha384-4UTPol6pUoBQFdmJuyiFgVA25Su8mEJu3iCvJrVvyU5XiHpslEt6nZ3IHfsnsvcQ es/languages/csp.min.js
sha384-SXCX0V+M0FVPrJEO9ZEpS3m9XNvb+6ri/SJ0PeYMbNCFYen+OSteL/zvqVwk3RCe es/languages/d.min.js
sha384-fzj3Z5pr5piPmIvzd2y0W0iL1hMnO4uDoFX4E0BiuFc6IX7/9GExg9z+rdtsa+HV es/languages/dart.min.js
sha384-w+JopPgc9Z2OK8qqUEep4trNAG6/xffK7ie6uRuh/Oobzf282Wv/Xmk+Ow1js03a es/languages/delphi.min.js
sha384-lYJsbJOBe66LVPGFLdfY5eH3UxyqlcHpLNaCqCD4qzpXcO6Uu7rVykUoKImpbHho es/languages/diff.min.js
sha384-rtkcZgHMYvJvBqj5I9tfAoP3HqFOrCQJTIBpqarlx6JuYNGml18bxqf+MHAPF3Xa es/languages/django.min.js
sha384-1EGAiVaUMgfNAzoLIDY7IMfvtuo6pwReZIEE+V6Xr9SutD+EO9kPDQDSKGxlUQgH es/languages/dns.min.js
sha384-y2/QSNZqtA/BScsNKoOFqPKYngPLmBdRNyrDD9qgwFoa8F6JTZipjArpeR7RwMqD es/languages/dockerfile.min.js
sha384-Jt+L9v2tvh5nUl5U83F/Z1V1yUIE6YKw1EWcUCk0qL0oG+ddcNJrI4lwQLJct+AE es/languages/dos.min.js
sha384-bZmBkKye1EhLscyGIurMoyEhNWUwjc43FfPPGfAOac7XmHp+KrwGDsMO1/HP9ial es/languages/ebnf.min.js
sha384-bV1W8gLIMo4pFF6nI59UKdZJ1FDtON9+wv8MRXJZr44EcjE8zj32nlB91EwngsCI es/languages/elixir.min.js
sha384-Nkd8VivENDot3hCzvDIAw3QFqAOD7iURphnS/kqe5khp6a4MZAGJ1mJG5JTVYNe/ es/languages/elm.min.js
sha384-TheZM55bmZYetXUKSd/CRQwu9+3XiSO7MgSVnDmwORQwSDIJ6B9+QdnhklCIEmva es/languages/erb.min.js
sha384-JXVG8/Oq7OEmKGcRuDij2/W87ucVLKUY4SdWMA+StcFA0/P/vS9k4Os6zLmzRcXv es/languages/erlang-repl.min.js
sha384-CRejkq1hqDWs3HoKEJejYyHZEVvAVs3h5CRZNU8BsaWlh4IAwvBN6QwvRuUPRyz3 es/languages/erlang.min.js
sha384-sSzz0Ej0VM9V2jE34Lw4Yv/eZhgtfZATAFJ+elLe9oFzf+FVimV31sCkyAtPapFA es/languages/excel.min.js
sha384-yx+/R9gXbmca4ORPO0PcEOXmhGVME/PfWd7Px3OL68qI2enbI5nHuz8IW1I54otf es/languages/fix.min.js
sha384-DNqKyQ2BKdgh5Q1FQQkBikiTwSwZytNoOcfIb4OdO+rt6GLdou70xaANXIMrglqm es/languages/flix.min.js
sha384-HwxQ16rZD0iMTrO4Y3ZeKgdpXBlun2o45gxAihf0PUAJX3NI1Hh1LyJ632onMGOR es/languages/fortran.min.js
sha384-V79Agif3kcKn9uJPHPN9pxEeRg4QN0vkDXcyixKMDzcOByrrJ9YtugJY+cF/ckJB es/languages/gams.min.js
sha384-7v5jXr+MUN2IxVYvvNOusmUpL2p8wtZaIRRbDoeojX4bv9QuTkFLCNYX19ieBv+a es/languages/gcode.min.js
sha384-ReUzUUctezGgHVHLXOBJDVhm8ncAr7CdQvmiN7meP2F0Xy4iec7mW2WSpDDLtHOM es/languages/gherkin.min.js
sha384-i/w8jIwL1y7lul32cOeW55d+LFRZokgXqjYOF6YTxdjBpZG+TxIdsAb4N/Mqz3AN es/languages/glsl.min.js
sha384-0Lge8Q3gkPiBmuhQWo/NODLTQHdLAVyld1ZWuffsTUiljg4RIliSu2ngdo8+PiAI es/languages/go.min.js
sha384-vB9hdQF82d7MA6UgU+A0fquNLIziuxNEn/GvbIIsFcNvQxADTbgai68aM31a2wWS es/languages/golo.min.js
sha384-vexIRfvWosLqv8JLeCY4tJbRjlWChuM9p5Ycf1wgAbzI4XQWBynryd9HHctcKULh es/languages/gradle.min.js
sha384-sPllOhz+hk+PxPPGdR0UeCDDWAhzwHtoXpwa4OchHD88pzBU90PRmYzWyLt9BrTS es/languages/graphql.min.js
sha384-PezHpZfoyYY8BmgOV3ogy1rwuAjOcGMiA4AtUs3i6iZ7m2iukOrqcAOwTIyCdi7i es/languages/haml.min.js
sha384-8C9A1UdPB1UZxa4JoUyMgGPQ+yT56Z6LY5dZfP4KqGYZ0yB+3e/PEivqB6wbzSgW es/languages/groovy.min.js
sha384-4hXDtv63cFCXp8rvXCl2I7ATX+hDt/y7D3FmD4lJamG/+8G8P9sGnWq4BL+EwXjp es/languages/handlebars.min.js
sha384-y84R3YozQxGbaQ0zilF8loQjKQrfwMG68fV47vn7E7qzJXpKCr1XoOTVy+C4at6n es/languages/haskell.min.js
sha384-7ktciqOF8h74zHArYMA5EbW0NhNqQ75TiUpUYkVok6VxlzvNDNc9bReB0cNKe3aM es/languages/haxe.min.js
sha384-ike1K58I73YYGNIywLgawq2IFWOv0tDVU1bU2LutxnNTJCADtBXv/BiiuUxJMyN6 es/languages/hsp.min.js
sha384-4Lqea1o3zRErOVpcJYIEx/xtde5rkKlK0RBdybdbuxSKQUq9ZJESoO98Nfo/a5ow es/languages/http.min.js
sha384-eIkJQpwld1OU3UbV7Ky1Cc2lG8NaQWnyXtJnVZI8e5gtafTQsqv0RPi+30ziUsqw es/languages/hy.min.js
sha384-9WZiSHGIeOEMyEXAPa+BHqfTpPFnSkpVW8fWAVehb7vGsws8hyhRBq6AXG8/7U1h es/languages/inform7.min.js
sha384-oMkqNgiD7maKde9spvMbhKMmVAfECvvK3LEM6lsApNnbuoAPjMfhgz9HD6dfusa+ es/languages/ini.min.js
sha384-ycTBw69hkkwdRlrN+3IRBjXsLnxOgSeT0P2A28GpunKEuhAQ3VTUEe89xm6FtCOP es/languages/irpf90.min.js
sha384-cJKdo+f0Qee/nQc4XreI9oTGjNsK6yenKaTWR06zv8T+nfwCr23HLMmUGLUGp4h6 es/languages/jboss-cli.min.js
sha384-9ZfhbXgA8vPWk9u4pmZiytLc1Ix11GBq6usHwUVebEYYQJRiqxx38kfAIAFFkEIn es/languages/json.min.js
sha384-8/7Zltp3QwdcNfuvEcsBUrffH/slowY7YTcFMmCoLuXkyiX/QwYY3QH50Q0A/SuE es/languages/julia-repl.min.js
sha384-2+3fEJ5fgBM4h6sfP9/JeOY842MFETt8SeoYOnX2hcqzeGvenx/Kr5JaZuu/D+bM es/languages/julia.min.js
sha384-Nj85v4unxXYbIL6LlsuKn9zmAlvTIhwhfTXjqvZ8mTh2pOCcSTISDN+y3asQeVGz es/languages/lasso.min.js
sha384-9/vyrBop3xK0RHhGjueybwI2CDCOaW78M/BuRDA7fxYqUAoKKT84fF6z9feE1eTU es/languages/latex.min.js
sha384-nJKRfyEu3Nzfdk7vaqfPKjRVMZ9cZl2iaxjCBUfIMk9GKaS1u7SQnpMKmeYdpRGO es/languages/ldif.min.js
sha384-vZJc4BJdtqiPTIeKWnFlMnzLlh48ECNolrjS1b6Kkn1vToLOt3LIpxJOMlp9pB46 es/languages/leaf.min.js
sha384-cG4BE6FOyg9J2arT2vlHN5WukF5ZjtE3w8hxxLrRgJ6cgFmnnWruSDateGPS++A5 es/languages/lisp.min.js
sha384-WZhqUAgF6Wo7e9jczjwJ1xsADXYBW0ZxpiYXn/HrR7t3EATuGQdy9qakjyC6FjO7 es/languages/livecodeserver.min.js
sha384-NDSxxSCM6xDtk2ny5V99eH4JAFJy1jJsETYZ34eKya37YzqePP7xrCeF631Nqgiv es/languages/lsl.min.js
sha384-Ejz9/vq5IVBe1c0yMGdC1J/xtqtyk7NMErrSttqMLsqA1A8Q5M8S2/wAZx9QY06J es/languages/llvm.min.js
sha384-aI4PwQ6eiaDle9o03RnayGxEh+1FvH8bGo6whcYufMNELqbc9vfIaHAkskI3hYOp es/languages/lua.min.js
sha384-uHVggrBycnxX7cJApYOHiDPUkW0cmAKFyGyDJpTQCJzLRs8IScvKMz4vKz1Yqviq es/languages/makefile.min.js
sha384-X/q41I88n7UzV97QlEeVU0gwfXOYCRA3lxbHRe+RwQagcu5JiKNn4tJUnJuC8aRN es/languages/markdown.min.js
sha384-34SS73BprmFLi50ALFwhnv9AT1hN9A5Uhz4DLJtW6wLL4r123k8sDecVQ5QY1nAa es/languages/matlab.min.js
sha384-UpID2eH8ewtRWfZXfTCD1yyFA0uQ57sgT+cgXI02G4rSCYEdCjteRMhf+FVjPjyd es/languages/mercury.min.js
sha384-smmBbm6rK5FCSE7vuOEnekOiQhmVz8/65q0Lpitex8B5m/wQIeh5EpQai1T4p7ID es/languages/mipsasm.min.js
sha384-MMJTTzUm4+COl8xDdh8LPv4ouf+a94Yj2+YARNlSQuiPYowdHE1KiU+ZCDBgRROB es/languages/mizar.min.js
sha384-hhckrsGplr4Vu+QLd+prj1U1WP6LZMo4cF6fTfrqGCoppdISqesohPEwY5ccjZt6 es/languages/mojolicious.min.js
sha384-lQCnwzigg85/npBmRflR9enDcYGso+Lvl8QTkF53DQpmdtN5D+gKXEGG7w0GwCyn es/languages/moonscript.min.js
sha384-Ap+gn8a9hJD0HZNYt5LfSLwsdBMs+qelf+Wk9SEOlkqnK3T3X0sCkAV0ICuUgSym es/languages/monkey.min.js
sha384-qs09e6bDAWFlmgP2gGBdCfypP24R1cuYoYHxAoWH796PsIApzNjQWap60ad/Wle9 es/languages/n1ql.min.js
sha384-KqzatixlwHuiVivw+OSe7j9Uorqy69jrfvnxVod+DiyMtlr+tcW10y3w1OxJPg93 es/languages/nestedtext.min.js
sha384-NnrmiTCJpdlvqWmOEAJmOnjJt3raR1bbHVRKrTf86uX+5+B6lLg1n1bNb9EFOkOL es/languages/nim.min.js
sha384-cqqhYpViplWKcCJOW6HrBEBDsEuud7lb8FSecmVxyjeXn6dLKqz2/rJZClBUBp0Q es/languages/nix.min.js
sha384-KKtDny6dj4GbvhT3fQIIrRYf12NDnI66xBJk332eoNIfdPbrkrkegOHaXOeR+nt8 es/languages/node-repl.min.js
sha384-hs2LkYUhDao6UzxI7t3F9JG7F9AdWDTSrMO/A+5tcvQQgN9AicfYeY8abXhRJSv5 es/languages/nginx.min.js
sha384-y1ZBkK7AAraPOYCIXBrdsd3V4MO2QI5keBAv4op1jTIZ8QyYRo2XMB9ybJF7rKEi es/languages/objectivec.min.js
sha384-zXRx/aN9MVz0q5SwlnvV4PXPsF/PfWQ1JCuxiBGq3nP0Mi24fcJxmsGAUxRCHKAZ es/languages/ocaml.min.js
sha384-HkwOWx30qOEIxMGrv1DjJICzNLlx4WxWC97ZQ9G8YNfuuhK74JpARWlZjhwIHnj7 es/languages/openscad.min.js
sha384-EPCdVMdNbPmsSxy4+4wgGOGCE9s2pRyZgBCM5RslUM+KJhQu1gZek05ligvvdJM8 es/languages/oxygene.min.js
sha384-JX9X3rNfFRgrBGJpUutvmR4kFE+pYNghKRmkHfY9UXQZ4Lazb2v4DRvCjUpLE3BY es/languages/pf.min.js
sha384-SN5YOzZddPoL7DDVdCvvVvHHpDsfZp6Xj0zyiRWmSN87IJCAoc8Np0SJpsujn0j0 es/languages/perl.min.js
sha384-gQafFuXFGSlfvsjDQAH4TW/pE9Q3shOSfukVGZdQYqWb7RMUrux/LXUtZq21kOT3 es/languages/php-template.min.js
sha384-Lbbz+EIU4FttO2f9X/EmVOP/xCeTLUgf0Fs5ONsR6U3/TbfYiNyFoQ2khqmcaJpk es/languages/php.min.js
sha384-2aCy6JEFB3d1Op03J8xDb0GTVSXhKaEO6ghqrzVSx8hNBC67FuzIZgp8N8xNuquQ es/languages/plaintext.min.js
sha384-DqKDH8RpybmViHnhVr4pfdEqJgfSBvqlY3qQiu/GcfFZCjK5BFj976q2otlEvRhR es/languages/pony.min.js
sha384-DJeYPUiASi7C/5CaMsM7SC83xjI0mp/QfWmWJrBJUFldEU9/F90i4Q2fOTsmiCz6 es/languages/powershell.min.js
sha384-wSJFkmjffdbicOMbCVvu+nWUYBCAv47uTgvaDLBlCiFdi4GWZAkdbNEGhQW6KUo9 es/languages/processing.min.js
sha384-bTSHV25LMEcQDFsT+WWPWGILh9yhhNwLRRY+5GisNEdb+TKdP+8eHWdL3+YiYXga es/languages/profile.min.js
sha384-4rrh2cnHt82evT0SEm3wX1cxGzKSaUwAj+CdaYj6gkn63TB3WKV8mL8AHsn9O80Q es/languages/prolog.min.js
sha384-f0d+TaVA00gm8NSV9IzFmRByfM9RY+K3tjyJUWkX2tkfDKBOOUNnQKBS4fl9pgRd es/languages/properties.min.js
sha384-Sggw92mmlLnaFLurzLirGOFxa4++VRMLLfBVj2gL+ojOwt1JRk93JBaPDHu7zKHm es/languages/protobuf.min.js
sha384-j40J5Vjx7Cni9YYZuTgGxSeWTAn60X8xzEmaNr21j/OMkC5AkqIyMOuO0Obi0a8v es/languages/puppet.min.js
sha384-mPmEpu5ThWEpzYnZdNba/WFR1w4wzFRF60GogHcscCW7+ohQN6K4LY2LMM9NyP87 es/languages/purebasic.min.js
sha384-2DRJBl02JqUsTErLRZ818FPi7HB0KjwPIbnhx8APwrNn3BP1HoJYkBizf/tA0sEb es/languages/python-repl.min.js
sha384-H58YuejjFvByN7GBO5TzbhtxoqyTZ3dTARHeyo5Ymmddx803bZcAp4w9/LigrO85 es/languages/python.min.js
sha384-8on3n+YbkiCq50MQa137oQWRnSUGkXJJOMCPNGwKS3XT/JK8zxIfqBoDohuh5/sN es/languages/q.min.js
sha384-c7enNWes8jS4Nk/hHdGhONhgBG0EGgj/r07rVXz+gIja2Z3SofzJLxsJ1Le9GWsp es/languages/qml.min.js
sha384-CQRTJUVtgT8MNoYGtq9Q7P0eBVtI+MKQTU+yu5w0HqibyQnEnBJG2Gzm1iWPO9rT es/languages/r.min.js
sha384-oulxy4SGPkrRRv3hH82ysn8cyrTNJqoLUrTy1e8Rp7gXH5FBP2GY7nbnNLkplZjj es/languages/reasonml.min.js
sha384-JxRN1f8gKhLZi4Ap+f06XZauVUeKYjPrgM1NrbPU8PO+jdFI6ZWVi2H7yeTfeASO es/languages/rib.min.js
sha384-amC1eGC6/nsz2naWi52p72S3yNdsJPGlPi5SFg4nEnjFu8Z1/oLgHJzZpEewseWD es/languages/roboconf.min.js
sha384-Vj+Z1ppnal9F/7tvbrRBBvbmq97Uv7910HO9MXUENunE6kxa4kf98Q3ilmJpUWoR es/languages/routeros.min.js
sha384-YWG+hxV+Iy8kTFiUnh4vzsNnW1I2V3YSK0TL1y0n9Md/uFejdP88qMbxC6BgbmNK es/languages/ruby.min.js
sha384-I5kO/USYfDQ/YpcBjOhkcvaLqGNcw+82tOPCKQg+pJ8vEaI2OblcXweIhm69cd6y es/languages/rsl.min.js
sha384-SEeJ+PE3UjD5JAP7Sw0ey+qYisQz1DMfKj8ycPAOuhDrTR4QUhnxIwPP1qyVlrsO es/languages/ruleslanguage.min.js
sha384-Xi59t57tUbNbtQmmE7RyQkZdPYsS7XA5LHJ++qw7blbxV33jEK9YWI/7bLsEDHQO es/languages/rust.min.js
sha384-QNiSqXQF/ZJSJzKjFZ2bt8sxiv+pm8VowT+dBFpaHJ6uFXJzL3nsggleGrCQKJVc es/languages/sas.min.js
sha384-COD94SZIFgvlee7p0jZxyqLGnYtdTPldPXqLknLf3W5vthJC6wj3LoEhYaz0eDC9 es/languages/scala.min.js
sha384-q5vX7uxR62gUcw2HiNbNMC2tg6zg69XZfpi3ZPErgqEuzm8IyizcPZAagRt1fZse es/languages/scheme.min.js
sha384-45a7EF00UZMeL4awSGt5p7oQB63Jwm90VwmCkPPOQWyhOdYI3FRQSUbATPu+UHd4 es/languages/scilab.min.js
sha384-KkHVBSmSSGZH701XFNTSEmi9W8ujN62BraqfdoqGxPBnNHEDFXhMas/d0cfvy0iR es/languages/shell.min.js
sha384-5BJ/l2aC3AtiacQFB/X2J4Py5H7+pangA1lRtdC1o0/IDk41qUCPNnvU0sDIBX5z es/languages/smali.min.js
sha384-dWZLvZGnAnLgfpfAgTayFBlbQ2MoQkoY86M5TNj2K07rvvP7x/V1f34vMh7/a7fL es/languages/smalltalk.min.js
sha384-5z02e2BLhrSgUcDKV2IS2a9zi3jI/fwpJrJUqET//f8H9LPNawUGD8sAOhGSoTum es/languages/sml.min.js
sha384-3JaZB8Lp0SXb7yHwg4O1QRvP3VYYr2OkjkuJVXPugEgVbQiuCtNQhi/fJ13tM25T languages/sqf.min.js
sha384-ccdC6+PVDumPW/ZCD3n48SkrjqPX626DFPWkOldfb0RzHb1ONezH/vstAnpDQbXK es/languages/sql.min.js
sha384-YTN2E/G8UaiUOg58dxX2k+psYz6pI69/tBarOxOKEstbO+DihkxT3Kn3MC7YgH7f es/languages/stan.min.js
sha384-i/1X57RE4uhzet0WPa/VSHFJZQCU7IXv2okh/+irdT3DrTKtQv7dJRp+Fcw9HuCt es/languages/step21.min.js
sha384-d7WYFjcZ5llVMNyqSEMeife4j8vp5r2HGT59chcdMTT3nXK6RKBA5rqdkjJoDvJ7 es/languages/subunit.min.js
sha384-kHt1/Uni78mSvQIfuxj5UAwYhcEAXnqRSis3NWs4yWzaumfocziBUXY1QT6Hb5bg es/languages/taggerscript.min.js
sha384-H4+K0a/IaEzJwDjpEtcNM4wbbZFCosAc824mg9MLlnu8gKRLE11w7EPS4ksYZ6sd es/languages/tap.min.js
sha384-jvMObKy0OrikQOJAQaI74rvT0Plz5Z4Plm46CxsF7ZX6n7XkEOQQ3I3pe2OKWPft es/languages/tcl.min.js
sha384-AX3CUFbxsP34rQoveSVvdrBzTFDi+0TYL1ElyqLhmcVd6Rf83OJBLZYIMpKhD7my es/languages/thrift.min.js
sha384-N/iKLW6WLSqpfqhClJtZwk8yEhXwEoTjeaJEn5X3E4+0vqdOA7s6S32+va4wZWbQ es/languages/twig.min.js
sha384-E1L2lprodexj8AfaNwZ4/9DJy2T518/Z0bm/OSxl1ja6tCRnfnlzRU4yCBy7ZgHD es/languages/tp.min.js
sha384-z/T3scOdptv4bY4P2nkmaLzLSqwc4G/LkizqUKyUkZUsThqgQPl52PsjTyi2IlEj es/languages/vala.min.js
sha384-261r+vQRe3NeUKaE1UWEWBTJoqrlyMp/ScTfQ0B7x6OmDWIf1niJBX6UFdBppj5b es/languages/vbnet.min.js
sha384-QpxCNuC27DSI513sK8dP8jAtN3ELbKUP1hHc4XNKFzxHSeiZ+x6XozLw085Wmskr es/languages/vbscript-html.min.js
sha384-fIfeVffAca1NwFefUmM1E1zlLDB1l7vQJ3dF4UG868k+GVE0tH9A+otPV25jkN9p es/languages/vbscript.min.js
sha384-xadZ9Mwk5iciGQeTyxIhA33ONsP/GYQvJOU7rbkBcvYDSaCd3cpG6O/YkBOyZYgE es/languages/verilog.min.js
sha384-0QgExm+Ma+7MziryGCiLLyzbQicUphMmdpNY1P+akukzyfcR8sbJ6Jqs1q63ft5V es/languages/vhdl.min.js
sha384-Tfu4joDYeyCsJG6qQJPLdt3vuYBClbZUpA8EAARQyFBqYpRQcOOqKnwEBDPgDJfy es/languages/vim.min.js
sha384-PJakpvQdF38W0l9QveQGFLnTuwL9PX+0lYzfHcKR53gWpYGaFLTcy5w73BZ4l0We es/languages/wasm.min.js
sha384-XsRSyIJyYdKZaY5XQCdW3VclDwDQAcQPv3H+lhRuJcR2HLf2sTdpNikiU2yWCnoh es/languages/wren.min.js
sha384-cxEm9CBaIkJMTP06i5/fPfZBG17++93Q/CPDic+dTl1i89WcfVds33zbPqC9emPS es/languages/xl.min.js
sha384-9oHHKuuGMcCMQzZJQ7DQrVFYtDK2psO++Cw1/yZpVmyKDPo7eC9AIP93qKMGHsMn es/languages/xml.min.js
sha384-QQFS11Zpe62si5zjtoA/fqQPRJsx2btCZCBS2BW2wkBztONZtxAjSnG4MEecEFY7 es/languages/xquery.min.js
sha384-HF0nYRdx/g1FMPCrLNc1F4VNrkyvsnpMhK43cxH6whjIudKpH1ItKUSjyiIOmsoT es/languages/yaml.min.js
sha384-2Ph73nc0HvX45JpBshrqyMfzuUxi0GbccDJzcNkolim16vi04fNGiLqiFMHqZNbx es/languages/zephir.min.js
sha384-JuxSzs6faTTonuuhril+9BnzR4YE71Fyaqd92YXkvxW0YZJCLNMjdKoqm65aORqu es/languages/parser3.min.js
sha384-u4ACJ6zRN8SRgBH+BNthfTMbX2c8VM6IC5MwQPRE4RysX/UAb0Y0sYP4pt2+LUpc es/languages/gauss.min.js
sha384-b44FXRWvvz69yR6UjtsJBJOcP5djhcFNbu00TOxJ8WtKD9BzqJYxR9YIgKMpB4+f languages/gml.min.js
sha384-qwWvi/gEMfSOYVEut4c75Fsk0WggqxWEzk/NGkQruRB2+2U2avMW4B9p9tKSXYwg es/languages/dsconfig.min.js
sha384-gS4VVORSHk+hbPy8woW/xJgN0yFZSyE85SufDOoGA+yzyLa7IolNvLmhH6Nxo4DA es/languages/dust.min.js
sha384-9LOwW7vb8GRsFc5SdTcK+jmF/VcUwxl7OFnT+UKYo7o5ZeQVwW4rKLbazhz1HDs4 es/languages/dts.min.js
sha384-U5kCdehnrAfJUqNpc6dCUTFb0im3cbKbuUscG4gBAs4oy6nHvxfJ4GYZFc8o1j3J es/languages/mel.min.js
sha384-FY8b6KC6C18+845vKwtaLK1J/TJ/b0Pk1YrsO9HGvQTUMKF8EUc1RdMKQ0+wIYGU es/languages/pgsql.min.js
sha384-YdUURwcSdcrMdraxWwYLFi5uxVo56AAdfT3oCKdzOeAvricXoUqaGiWjJwtILV3p es/languages/stata.min.js
sha384-VyT4rDIU6LcfvnTFX+TGqSTEyNV91e75L1vq6uOiMoDnX5SAtH64f+NunSm1geNv es/languages/x86asm.min.js
sha384-gZBE9LjpYxATv7ItxU2WYQ0YLJ3dMenrmwqvKX74DRr+hP2fFwELPTlYSd566GYp languages/isbl.min.js
sha384-ucy+EEMr3dudT3OqvK1wtpEpsDlSO6YjJQPIKg4M5MHZbfQPTe/XG5aXh34ykGAG es/languages/maxima.min.js
sha384-lU0G3BUDMVkQ/j4bFuzpzeoSTGRco1KOcA7XhL9ixztdKRPCU2eId6BRymwzJ4y6 es/languages/sqf.min.js
sha384-m6+jwMXWldke74+g00kdA05S0T2BTK5HLU84vimZYbFh62Y+hJc3AfMCkw9xg814 es/languages/gml.min.js
sha384-OmsG99ZoimTYVAkGXKurzPJV9nOupPq6XZovMrQmO54/Zfc33lhZWaFmdU5lIO7d es/languages/isbl.min.js
sha384-ZeRlL93a14C9TpOteJYeBNumAy4CGJrYLEuXavIbTN6Jl6f98SiCD2+5kbhpscOf es/languages/1c.min.js
sha384-7ArLU+f1/qhVVHcOZ0ldkM00kgred5WTPVUY/yvEsL5MzXZ+yDPhLW5DF2zZjEjs languages/arduino.min.js
sha384-uvUMwnqJnWLAIxByubckbmzV9CuBr1YjlChk9VH9fftWJUed5d1BYHJOuWMioSfL languages/fsharp.min.js
sha384-Tk/yCllhl9WTsUacwGrZjVCf8rE6s2Uf0yTp1jCTxJB1pXkizIAT2G+emrx9lM/o languages/nsis.min.js
sha384-iz1efG+VF5JrS8dB3+DM5S8ZtEp0pzDOifjmmnEomsxzzTS4MMwNoIKVfWhyS/+V languages/coffeescript.min.js
sha384-SHhXPCX2gNLAr3g7J1jizp2ogvCmt9G7x10X+J3ELuJ3XUGPh8a+cmO+yVeyczEl languages/javascript.min.js
sha384-Zvx+esgh/ZQ1RvphnxDMo0DnGF5DekimKZf/s51FUNvpTbqSh9XWN9VzOsmMZoW/ languages/livescript.min.js
sha384-2NV0rAxtYJUgIdvI8AGKajds0aR9CzQr2Lx/3TewUnieTuja7qiech01s99OKAEe languages/css.min.js
sha384-Y6s90N8qvj1iBkanV26o+AMS/2LhYAYu3KX+hIaSsCRcNxs+dSrrHLvDZhkCoPui languages/less.min.js
sha384-kBUxybeDUzNBI9qPb2siVEodTB87o2DTBT0F0CQDly5vl7ixQcTGaleGKv2z2NME languages/scss.min.js
sha384-iC3jo8aXHggvBGIjZn1cvumecXwko8ogQpL2xPGa0b5mxDVuGFcNy2aNoW69C4MU languages/stylus.min.js
sha384-ppS3kmYuIwUHk7ZpUIgfhshP0bdLTCuGZfTXAsF8MNaTCsTGSj7IKqFGm9ERyTMd languages/java.min.js
sha384-m7s7hvqy10rUilBF+4c5S7SNeBcBU9Nk8biWVpO2K3qIoBL10MH+kTCpUja8eNUa languages/kotlin.min.js
sha384-G0QnP9AoA9z0DiHjDtwLS3SPr9OsLAdX9NLvg8FFvq4zmchjcNq1G5Z8p8fV7GMJ es/languages/arduino.min.js
sha384-bCKSkbIWIp32toyiikgpZwSMDt3f1HkJ+1D5aPSBgeouJjAedFdJ2bFKKW5rqj8G es/languages/fsharp.min.js
sha384-G/93J76v5ZhJsLSaAyhRGmVkqopgDDNYZx/5Afw28/wPH8vVcWCnPbmvgDkoun0u es/languages/nsis.min.js
sha384-5z5Wqg0v+aDRGtE8ql0qTiDCaFsfdUT2fyjVKjfefLswwbvL8jEYmZBnnR0z0VN3 es/languages/coffeescript.min.js
sha384-DyMEwt9N7i4S+GRUmuxEyx9kyJOqY/NUQWbhhjIF499xyeWlBJnZeaywNHJSMWvn es/languages/javascript.min.js
sha384-uV0va95RrySFmXJR6pAYDwvCbJvk50RXH6hjlm0DHlnF5penjRRzUC3A5jKWLh4n es/languages/livescript.min.js
sha384-XgyICbHI8NcqKGQUkgZHzCk0rYs85wUm6g5MjYo1KUod07OB6C1cwyMq/ss/NcVc es/languages/css.min.js
sha384-jeldGWVyxlorjExh0wwW0rh1I552Y8TMp276NF/u1J62ym/bdsCaAYEJ3Nz2gSg/ es/languages/less.min.js
sha384-4aYZOVxsIWfwuSPvYSRPnVUd+88iLI9ZVqwJjgy0Xkh/9fMvivrj4i3fDRUiBs6k es/languages/scss.min.js
sha384-977g5DQ0UC5VYnnaaLubqsMux9ogyZh2Nv1JjXwqoqVqva8G33tstStuNq5GmqQw es/languages/stylus.min.js
sha384-wTaTo349vOZe72ENFmj9KmmorH0TZkS4pmHL2YYDVcFAxr7f9f9iv3qFLMYw9X4i es/languages/java.min.js
sha384-EsCFk4eb0UjEZ26uSbSrgwkLD2oT2iGwGWpYr3AFiw3eWcF1inTwVzTdYYvtnyum es/languages/kotlin.min.js
sha384-f4paDAxciCRNi20wvqsw+cRLJGMzqL6ACSmflRupuGR+iO14uu2Jq1l/jQcuwrvP languages/mathematica.min.js
sha384-2P2JCH4MG/wqE0WvIxKmjAPI7CoJ4oNEZTAF2vWBsRI9KpwXqCoeAquUC293Ax8F es/languages/mathematica.min.js
sha384-HEEMkUmM4Qq79lFD+lW07y+65uimb0naW+pg+F3yyZWxRa4Ph+I7KB6sFlpu+uGj languages/swift.min.js
sha384-DYoPlCDcXrP3/Twe76kF0hp72pGP0/2BLvB965FR08PxB8fhoLu3lp4OM+IwmitH languages/typescript.min.js
sha384-bPmt8yGeGWKl7WZeNwdw3ah1Sa7PrikWpK36KBv5P0Lj7F0yKvAcFLwWnziKwPJb es/languages/swift.min.js
sha384-EOKjwxtJpPTT2PGJ2whEV3/798as+mCdVqZrIpFpI1SBcVY705sSizRzyjq06wNl es/languages/typescript.min.js
```
