<?php

namespace nordmind\UserBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use FOS\UserBundle\Model\User as BaseUser;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use FOS\MessageBundle\Model\ParticipantInterface;
use nordmind\AppBundle\Utility;
use nordmind\ArtistBundle\Entity\Category;
use nordmind\ArtistBundle\Entity\Photo;
use Doctrine\Common\Collections\Criteria;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * @ORM\Entity
 * @ORM\HasLifecycleCallbacks()
 * @Gedmo\SoftDeleteable(fieldName="deletedAt", timeAware=false, hardDelete=false)
 * @ORM\Entity(repositoryClass="nordmind\UserBundle\Entity\UserRepository")
 * @ORM\Table(name="select_fos_user")
 */
class User extends BaseUser implements ParticipantInterface
{
    const ROLE_ARTIST = 'ROLE_ARTIST';
    const ROLE_CLIENT = 'ROLE_CLIENT';
    const ROLE_USER = 'ROLE_USER';
    const ROLE_ADMIN = 'ROLE_ADMIN';
    const ROLE_SUPER_ADMIN = 'ROLE_SUPER_ADMIN';
    const TYPE_ARTIST_WITH_AN_AGENCY = 1;
    const TYPE_FREELANCER = 2;
    const TYPE_AGENCY = 11;
    const TYPE_MEDIA = 12;
    const TYPE_MAGAZINE = 13;
    const TYPE_PRODUCER = 14;
    const PROFESSION_PHOTOGRAPHER = 1;
    const PROFESSION_VIDEOGRAPHER = 2;
    const PROFESSION_ILUSTRATOR = 3;

    /**
     * @ORM\Id
     * @ORM\Column(type="integer")
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    protected $id;

    public function __construct()
    {
        parent::__construct();
        $this->favorites = new ArrayCollection();
        $this->photos = new ArrayCollection();
    }

    /**
     *
     * @var string
     * @ORM\Column(name="name", type="string", length=512, nullable=false)
     *
     */
    private $name;

    /**
     *
     * @var string
     * @ORM\Column(name="last_name", type="string", length=512, nullable=true)
     *
     */
    private $lastName;

    /**
     * @Gedmo\Slug(fields={"name"}, updatable=false)
     * @ORM\Column(length=255, unique=true)
     */
    private $slug;

    /**
     *
     * @var string
     * @ORM\Column(name="company", type="string", length=512, nullable=true)
     *
     */
    private $company;

    /**
     *
     * @var string
     * @ORM\Column(name="vat", type="string", length=32, nullable=true)
     *
     */
    private $vat;

    /**
     *
     * @var string
     * @ORM\Column(name="city", type="string", length=512, nullable=true)
     *
     */
    private $city;

    /**
     *
     * @var string
     * @ORM\Column(name="street", type="string", length=512, nullable=true)
     *
     */
    private $street;

    /**
     *
     * @var string
     * @ORM\Column(name="zipCode", type="string", length=32, nullable=true)
     *
     */
    private $zipCode;

    /**
     *
     * @var string
     * @ORM\Column(name="country", type="string", length=256, nullable=true)
     *
     */
    private $country;

    /**
     *
     * @var string
     * @ORM\Column(name="phone", type="string", length=64, nullable=true)
     *
     */
    private $phone;

    /**
     *
     * @var string
     * @ORM\Column(name="website", type="string", length=512, nullable=true)
     *
     */
    private $website;

    /**
     *
     * @var string
     * @ORM\Column(name="facebook", type="string", length=512, nullable=true)
     *
     */
    private $facebook;

    /**
     *
     * @var string
     * @ORM\Column(name="instagram", type="string", length=512, nullable=true)
     *
     */
    private $instagram;

    /**
     * @var integer
     *
     * @ORM\Column(name="type", type="smallint", nullable=true)
     */
    protected $type;

    /**
     * @var integer
     *
     * @ORM\Column(name="profession", type="smallint", nullable=true)
     */
    protected $profession;

    /**
     *
     * @var boolean
     * @ORM\Column(name="has_logo", type="boolean")
     *
     */
    private $hasLogo = false;

    /**
     *
     * @var boolean
     * @ORM\Column(name="check_terms", type="boolean")
     *
     */
    private $checkTerms = false;

    /**
     *
     * @var boolean
     * @ORM\Column(name="check_newsletter", type="boolean")
     *
     */
    private $checkNewsletter = false;

    /**
     *
     * @var boolean
     * @ORM\Column(name="check_rodo", type="boolean")
     *
     */
    private $checkRodo = false;

    /**
     *
     * @var boolean
     * @ORM\Column(name="check_invoice", type="boolean")
     *
     */
    private $checkInvoice = false;

    /**
     * @var \DateTime $createdAt
     *
     * @Gedmo\Timestampable(on="create")
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $createdAt;

    /**
     * @var \DateTime $updatedAt
     *
     * @Gedmo\Timestampable(on="update")
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $updatedAt;

    /**
     * @var \DateTime $contentChanged
     *
     * @ORM\Column(name="content_changed_at", type="datetime", nullable=true)
     * @Gedmo\Timestampable(on="change", field={"name"})
     */
    private $contentChangedAt;

    /**
     * @ORM\Column(name="deleted_at", type="datetime", nullable=true)
     */
    private $deletedAt;

    /**
     * @var Photo
     *
     * @ORM\OneToMany(targetEntity="nordmind\ArtistBundle\Entity\Photo", mappedBy="user", cascade={"persist"})
     * @ORM\OrderBy({"position" = "ASC"})
     */
    private $photos;

    /**
     * @var ArrayCollection
     */
    private $uploadedPhotos;

    /**
     * @var Favorites
     *
     * @ORM\OneToMany(targetEntity="nordmind\ArtistBundle\Entity\Favorite", mappedBy="user", cascade={"persist"}, orphanRemoval=true)
     */
    private $favorites;

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @param string $name
     */
    public function setName($name)
    {
        $this->name = $name;
    }

    /**
     * @return string
     */
    public function getLastName()
    {
        return $this->lastName;
    }

    /**
     * @param string $lastName
     */
    public function setLastName($lastName)
    {
        $this->lastName = $lastName;
    }

    public function getFullName()
    {
        return $this->getName();
    }

    function getPhone()
    {
        return $this->phone;
    }

    function setPhone($phone)
    {
        $this->phone = $phone;
    }

    function getHasLogo()
    {
        return $this->hasLogo;
    }

    function setHasLogo($hasLogo)
    {
        $this->hasLogo = $hasLogo;
    }

    public function getRoleFormatted()
    {
        if (in_array(self::ROLE_SUPER_ADMIN, $this->getRoles())) {
            return "Superadministrator";
        } else if (in_array(self::ROLE_ADMIN, $this->getRoles())) {
            return "Administrator";
        } else if (in_array(self::ROLE_USER, $this->getRoles())) {
            return "User";
        }
    }

    public static function getRoleArray()
    {
        return array(self::ROLE_SUPER_ADMIN => "Superadministrator",
            self::ROLE_ADMIN => "Administrator",
            self::ROLE_USER => "User");
    }

    public static function getTypeArray()
    {
        return array(self::TYPE_ARTIST_WITH_AN_AGENCY => "Artist with an agency",
            self::TYPE_FREELANCER => "Freelancer",
            self::TYPE_AGENCY => "Agency",
            self::TYPE_MEDIA => 'Media',
            self::TYPE_MAGAZINE => 'Magazine',
            self::TYPE_PRODUCER => 'Producer'
        );
    }

    public static function getArtistTypeArray()
    {
        return array(self::TYPE_ARTIST_WITH_AN_AGENCY => "Artist with an agency",
            self::TYPE_FREELANCER => "Freelancer"
        );
    }

    public static function getClientTypeArray()
    {
        return array(
            self::TYPE_AGENCY => "Agency",
            self::TYPE_MEDIA => 'Media',
            self::TYPE_MAGAZINE => 'Magazine',
            self::TYPE_PRODUCER => 'Producer'
        );
    }

    public static function getProfessionArray(TranslatorInterface $translator)
    {
        return array(
            self::PROFESSION_PHOTOGRAPHER => $translator->trans("Photographer"),
            self::PROFESSION_VIDEOGRAPHER => $translator->trans("Videographer"),
            self::PROFESSION_ILUSTRATOR => $translator->trans("Ilustrator"),
        );
    }

    public function __toString()
    {
        return $this->getName();
    }

    public function getLogoSource()
    {
        return $this->getHasLogo() ? '/uploads/images/logo/' . $this->getId() . '.jpg' : '/images/adminbsb/user.png';
    }

    function getCreatedAt()
    {
        return $this->createdAt;
    }

    function getUpdatedAt()
    {
        return $this->updatedAt;
    }

    function getContentChangedAt()
    {
        return $this->contentChangedAt;
    }

    function getDeletedAt()
    {
        return $this->deletedAt;
    }

    function setCreatedAt(\DateTime $createdAt)
    {
        $this->createdAt = $createdAt;
    }

    function setUpdatedAt(\DateTime $updatedAt)
    {
        $this->updatedAt = $updatedAt;
    }

    function setContentChangedAt(\DateTime $contentChangedAt)
    {
        $this->contentChangedAt = $contentChangedAt;
    }

    function setDeletedAt($deletedAt)
    {
        $this->deletedAt = $deletedAt;
    }

    public function setEmail($email)
    {
        $email = is_null($email) ? '' : $email;
        parent::setEmail($email);
        $this->setUsername($email);

        return $this;
    }

    function getCompany()
    {
        return $this->company;
    }

    function getVat()
    {
        return $this->vat;
    }

    function getCity()
    {
        return $this->city;
    }

    function getStreet()
    {
        return $this->street;
    }

    function getZipCode()
    {
        return $this->zipCode;
    }

    function getCountry()
    {
        return $this->country;
    }

    function getType()
    {
        return $this->type;
    }

    function setCompany($company)
    {
        $this->company = $company;
    }

    function setVat($vat)
    {
        $this->vat = $vat;
    }

    function setCity($city)
    {
        $this->city = $city;
    }

    function setStreet($street)
    {
        $this->street = $street;
    }

    function setZipCode($zipCode)
    {
        $this->zipCode = $zipCode;
    }

    function setCountry($country)
    {
        $this->country = $country;
    }

    function setType($type)
    {
        $this->type = $type;
    }

    function getWebsite()
    {
        return $this->website;
    }

    function setWebsite($website)
    {
        $this->website = $website;
    }

    function getCheckTerms()
    {
        return $this->checkTerms;
    }

    function getCheckNewsletter()
    {
        return $this->checkNewsletter;
    }

    function getCheckRodo()
    {
        return $this->checkRodo;
    }

    function getCheckInvoice()
    {
        return $this->checkInvoice;
    }

    function setCheckTerms($checkTerms)
    {
        $this->checkTerms = $checkTerms;
    }

    function setCheckNewsletter($checkNewsletter)
    {
        $this->checkNewsletter = $checkNewsletter;
    }

    function setCheckRodo($checkRodo)
    {
        $this->checkRodo = $checkRodo;
    }

    function setCheckInvoice($checkInvoice)
    {
        $this->checkInvoice = $checkInvoice;
    }

    function getPhotos()
    {
        if ($this->getProfession() == self::PROFESSION_VIDEOGRAPHER)
        {
            return $this->photos->filter(function(Photo $photo) {
                return $photo->isVideo() == true;
            });
        }
        return $this->photos;
    }

    function getCategories()
    {
        $data = [];
        foreach ($this->photos as $photo)
        {
            $categories = $photo->getCategories();
            foreach ($categories as $category)
            {
                $data[$category->getCategory()] = Category::getByKey($category->getCategory());
            }
        }
        return $data;
    }

    function getUploadedPhotos()
    {
        return $this->uploadedPhotos;
    }

    function setPhotos(Photo $photos)
    {
        $this->photos = $photos;
    }

    function getFacebook()
    {
        return $this->facebook;
    }

    function getInstagram()
    {
        return $this->instagram;
    }

    function setFacebook($facebook)
    {
        $this->facebook = $facebook;
    }

    function setInstagram($instagram)
    {
        $this->instagram = $instagram;
    }

    function getProfession()
    {
        return $this->profession;
    }

    function setProfession($profession)
    {
        $this->profession = $profession;
    }

    function canUploadPhotos()
    {
        return $this->getProfession() == self::PROFESSION_PHOTOGRAPHER || $this->getProfession() == self::PROFESSION_ILUSTRATOR;
    }

    function canAddVideo()
    {
        return $this->getProfession() == self::PROFESSION_VIDEOGRAPHER;
    }

    /**
     * @return Favorites
     */
    public function getFavorites()
    {
        return $this->favorites;
    }

    /**
     * @return mixed
     */
    public function getSlug()
    {
        return $this->slug;
    }

    public function hasFavorite(Photo $photo)
    {
        $criteria = Criteria::create()->where(Criteria::expr()->eq("photo", $photo));
        return count($this->getFavorites()->matching($criteria)) > 0 ? true : false;
    }

    public function getCover()
    {
        $criteria = Criteria::create()
            ->where(Criteria::expr()->eq("isCover", true))
        ;

        $cover = $this->photos->matching($criteria);
        return $cover->first() ? $cover->first() : null;
    }


}
